/**
 * Parses CSV text into an array of rows, where each row is an array of cell values.
 * Handles quoted fields, newlines within quotes, and escaped quotes.
 * @param {string} text - The CSV/TSV text to parse.
 * @param {string} delimiter - The delimiter to use (default: auto-detect).
 * @returns {Array<Array<string>>}
 */
export const parseCSV = (text, delimiter = null) => {
    if (!text) return [];

    // Auto-detect delimiter if not provided (tab for clipboard, comma for files usually)
    if (!delimiter) {
        delimiter = text.includes('\t') ? '\t' : ',';
    }

    const rows = [];
    let currentRow = [];
    let currentCell = '';
    let insideQuotes = false;
    let i = 0;

    // Normalize newlines
    text = text.replace(/\r\n/g, '\n').replace(/\r/g, '\n');

    while (i < text.length) {
        const char = text[i];
        const nextChar = text[i + 1];

        if (char === '"') {
            if (insideQuotes && nextChar === '"') {
                // Escaped quote ("") -> treat as single quote inside cell
                currentCell += '"';
                i++; // Skip next quote
            } else {
                // Toggle quote state
                insideQuotes = !insideQuotes;
            }
        } else if (char === delimiter && !insideQuotes) {
            // End of cell
            currentRow.push(currentCell);
            currentCell = '';
        } else if (char === '\n' && !insideQuotes) {
            // End of row
            currentRow.push(currentCell);
            rows.push(currentRow);
            currentRow = [];
            currentCell = '';
        } else {
            // Regular character
            currentCell += char;
        }
        i++;
    }

    // Add last cell/row if any
    if (currentCell || currentRow.length > 0) {
        currentRow.push(currentCell);
        rows.push(currentRow);
    }

    // Filter empty trailing row often generated by clipboard
    if (rows.length > 0 && rows[rows.length - 1].length === 1 && rows[rows.length - 1][0] === '') {
        rows.pop();
    }

    return rows;
};

/**
 * Converts an array of objects to a CSV string.
 * @param {Array<Object>} data - The data to convert.
 * @param {Array<Object>} columns - Column definitions { key, label }.
 * @returns {string}
 */
export const generateCSV = (data, columns) => {
    // Header row
    const headers = columns.map(c => `"${c.label.replace(/"/g, '""')}"`).join(',');

    // Data rows
    const rows = data.map(item => {
        return columns.map(col => {
            let val = '';

            // Handle nested objects or direct properties
            if (col.isResult) {
                // For results object
                const keyMap = { 'intro': 'intro', 'content': 'content', 'faqs': 'faqs', 'meta': 'meta', 'title': 'title' };
                val = item.results?.[keyMap[col.key] || col.key] || '';
            } else {
                // For main object properties
                val = item[col.key] || '';
            }

            // Ensure value is string
            const stringVal = String(val);

            // Escape quotes and wrap in quotes if necessary
            if (stringVal.includes('"') || stringVal.includes(',') || stringVal.includes('\n')) {
                return `"${stringVal.replace(/"/g, '""')}"`;
            }
            return stringVal;
        }).join(',');
    });

    return [headers, ...rows].join('\n');
};
